// --- Game Config ---
const PLAYERS_IN_GAME = 6;

// --- Board Data (نسخه نهایی با خانه‌ها و جدول اجاره) ---
const boardSpaces = [
    { id: 0, name: "شروع", type: "go" },
    { id: 1, name: "خیابان جردن", type: "property", price: 60, owner: null, color: "brown", houses: 0, rent: [2, 10, 30, 90, 160, 250], house_cost: 50, mortgage_value: 30, mortgaged: false },
    { id: 2, name: "خزانه عمومی", type: "chest" },
    { id: 3, name: "خیابان ولیعصر", type: "property", price: 60, owner: null, color: "brown", houses: 0, rent: [4, 20, 60, 180, 320, 450], house_cost: 50, mortgage_value: 30, mortgaged: false },
    { id: 4, name: "مالیات بر درآمد", type: "tax", amount: 200 },
    { id: 5, name: "راه آهن شمال", type: "railroad", price: 200, owner: null, mortgage_value: 100, mortgaged: false },
    { id: 6, name: "خیابان مطهری", type: "property", price: 100, owner: null, color: "light-blue", houses: 0, rent: [6, 30, 90, 270, 400, 550], house_cost: 50, mortgage_value: 50, mortgaged: false },
    { id: 7, name: "کارت شانس", type: "chance" },
    { id: 8, name: "خیابان بهشتی", type: "property", price: 100, owner: null, color: "light-blue", houses: 0, rent: [6, 30, 90, 270, 400, 550], house_cost: 50, mortgage_value: 50, mortgaged: false },
    { id: 9, name: "خیابان سهروردی", type: "property", price: 120, owner: null, color: "light-blue", houses: 0, rent: [8, 40, 100, 300, 450, 600], house_cost: 50, mortgage_value: 60, mortgaged: false },
    { id: 10, name: "زندان", type: "jail" },
    { id: 11, name: "میدان ونک", type: "property", price: 140, owner: null, color: "pink", houses: 0, rent: [10, 50, 150, 450, 625, 750], house_cost: 100, mortgage_value: 70, mortgaged: false },
    { id: 12, name: "شرکت برق", type: "utility", price: 150, owner: null, mortgage_value: 75, mortgaged: false },
    { id: 13, name: "میدان تجریش", type: "property", price: 140, owner: null, color: "pink", houses: 0, rent: [10, 50, 150, 450, 625, 750], house_cost: 100, mortgage_value: 70, mortgaged: false },
    { id: 14, name: "میدان ولیعصر", type: "property", price: 160, owner: null, color: "pink", houses: 0, rent: [12, 60, 180, 500, 700, 900], house_cost: 100, mortgage_value: 80, mortgaged: false },
    { id: 15, name: "راه آهن غرب", type: "railroad", price: 200, owner: null, mortgage_value: 100, mortgaged: false },
    { id: 16, name: "بلوار کشاورز", type: "property", price: 180, owner: null, color: "orange", houses: 0, rent: [14, 70, 200, 550, 750, 950], house_cost: 100, mortgage_value: 90, mortgaged: false },
    { id: 17, name: "خزانه عمومی", type: "chest" },
    { id: 18, name: "بلوار میرداماد", type: "property", price: 180, owner: null, color: "orange", houses: 0, rent: [14, 70, 200, 550, 750, 950], house_cost: 100, mortgage_value: 90, mortgaged: false },
    { id: 19, name: "بلوار آفریقا", type: "property", price: 200, owner: null, color: "orange", houses: 0, rent: [16, 80, 220, 600, 800, 1000], house_cost: 100, mortgage_value: 100, mortgaged: false },
    { id: 20, name: "پارکینگ رایگان", type: "free-parking" },
    { id: 21, name: "خیابان پاسداران", type: "property", price: 220, owner: null, color: "red", houses: 0, rent: [18, 90, 250, 700, 875, 1050], house_cost: 150, mortgage_value: 110, mortgaged: false },
    { id: 22, name: "کارت شانس", type: "chance" },
    { id: 23, name: "خیابان شریعتی", type: "property", price: 220, owner: null, color: "red", houses: 0, rent: [18, 90, 250, 700, 875, 1050], house_cost: 150, mortgage_value: 110, mortgaged: false },
    { id: 24, name: "خیابان دولت", type: "property", price: 240, owner: null, color: "red", houses: 0, rent: [20, 100, 300, 750, 925, 1100], house_cost: 150, mortgage_value: 120, mortgaged: false },
    { id: 25, name: "راه آهن جنوب", type: "railroad", price: 200, owner: null, mortgage_value: 100, mortgaged: false },
    { id: 26, name: "میدان صنعت", type: "property", price: 260, owner: null, color: "yellow", houses: 0, rent: [22, 110, 330, 800, 975, 1150], house_cost: 150, mortgage_value: 130, mortgaged: false },
    { id: 27, name: "میدان کاج", type: "property", price: 260, owner: null, color: "yellow", houses: 0, rent: [22, 110, 330, 800, 975, 1150], house_cost: 150, mortgage_value: 130, mortgaged: false },
    { id: 28, name: "سازمان آب", type: "utility", price: 150, owner: null, mortgage_value: 75, mortgaged: false },
    { id: 29, name: "میدان سرو", type: "property", price: 280, owner: null, color: "yellow", houses: 0, rent: [24, 120, 360, 850, 1025, 1200], house_cost: 150, mortgage_value: 140, mortgaged: false },
    { id: 30, name: "برو به زندان", type: "go-to-jail" },
    { id: 31, name: "خیابان فرشته", type: "property", price: 300, owner: null, color: "green", houses: 0, rent: [26, 130, 390, 900, 1100, 1275], house_cost: 200, mortgage_value: 150, mortgaged: false },
    { id: 32, name: "خیابان الهیه", type: "property", price: 300, owner: null, color: "green", houses: 0, rent: [26, 130, 390, 900, 1100, 1275], house_cost: 200, mortgage_value: 150, mortgaged: false },
    { id: 33, name: "خزانه عمومی", type: "chest" },
    { id: 34, name: "خیابان زعفرانیه", type: "property", price: 320, owner: null, color: "green", houses: 0, rent: [28, 150, 450, 1000, 1200, 1400], house_cost: 200, mortgage_value: 160, mortgaged: false },
    { id: 35, name: "راه آهن شرق", type: "railroad", price: 200, owner: null, mortgage_value: 100, mortgaged: false },
    { id: 36, name: "کارت شانس", type: "chance" },
    { id: 37, name: "خیابان فرمانیه", type: "property", price: 350, owner: null, color: "dark-blue", houses: 0, rent: [35, 175, 500, 1100, 1300, 1500], house_cost: 200, mortgage_value: 175, mortgaged: false },
    { id: 38, name: "مالیات", type: "tax", amount: 100 },
    { id: 39, name: "خیابان نیاوران", type: "property", price: 400, owner: null, color: "dark-blue", houses: 0, rent: [50, 200, 600, 1400, 1700, 2000], house_cost: 200, mortgage_value: 200, mortgaged: false }
];

// --- Card Decks Data ---
const communityChestCards = [
    { text: "برو به زندان", instruction: "مستقیماً به زندان بروید، از خانه شروع عبور نکنید و ۲۰۰ دلار نگیرید.", action: "go_to_jail" },
    { text: "کارت آزادی از زندان", instruction: "این کارت را می‌توانید نگه دارید تا زمانی که به آن نیاز پیدا کنید یا آن را بفروشید.", action: "get_out_of_jail_free" },
    { text: "پیشروی به خانه شروع", instruction: "۲۰۰ دلار جایزه دریافت کنید.", action: "move_to", value: 0, collect_go: true },
    { text: "هزینه بیمارستان", instruction: "۱۰۰ دلار بپردازید.", action: "pay_money", value: 100 },
    { text: "هزینه دکتر", instruction: "۵۰ دلار بپردازید.", action: "pay_money", value: 50 },
    { text: "ارث", instruction: "۱۰۰ دلار دریافت کنید.", action: "get_money", value: 100 },
    { text: "فروش سهام", instruction: "از فروش سهام خود ۵۰ دلار دریافت می‌کنید.", action: "get_money", value: 50 },
    { text: "بیمه عمر", instruction: "بیمه عمر شما سررسید می‌شود. ۱۰۰ دلار دریافت کنید.", action: "get_money", value: 100 },
    { text: "صندوق تعطیلات", instruction: "صندوق تعطیلات شما سررسید می‌شود. ۱۰۰ دلار دریافت کنید.", action: "get_money", value: 100 },
    { text: "بازگشت مالیات", instruction: "۲۰ دلار دریافت کنید.", action: "get_money", value: 20 },
    { text: "جایزه زیبایی", instruction: "جایزه دوم مسابقه زیبایی. ۱۰ دلار دریافت کنید.", action: "get_money", value: 10 },
    { text: "هزینه مدرسه", instruction: "۵۰ دلار بپردازید.", action: "pay_money", value: 50 },
    { text: "خدمات مشاوره", instruction: "۲۵ دلار دریافت کنید.", action: "get_money", value: 25 },
    { text: "جریمه تعمیرات", instruction: "به ازای هر خانه ۴۰ دلار و به ازای هر هتل ۱۱۵ دلار بپردازید.", action: "street_repairs", house_cost: 40, hotel_cost: 115 },
    { text: "خطای بانکی", instruction: "خطای بانکی به نفع شما. ۲۰۰ دلار دریافت کنید.", action: "get_money", value: 200 },
    { text: "تولد", instruction: "تولد شماست. از هر بازیکن ۱۰ دلار دریافت کنید.", action: "get_from_players", value: 10 }
];

const chanceCards = [
    { text: "برو به زندان", instruction: "مستقیماً به زندان بروید، از خانه شروع عبور نکنید و ۲۰۰ دلار نگیرید.", action: "go_to_jail" },
    { text: "کارت آزادی از زندان", instruction: "این کارت را می‌توانید نگه دارید تا زمانی که به آن نیاز پیدا کنید یا آن را بفروشید.", action: "get_out_of_jail_free" },
    { text: "پیشروی به خانه شروع", instruction: "۲۰۰ دلار جایزه دریافت کنید.", action: "move_to", value: 0, collect_go: true },
    { text: "عقب‌گرد", instruction: "سه خانه به عقب برگردید.", action: "move_relative", value: -3 },
    { text: "پیشروی به خیابان دولت", instruction: "اگر از خانه شروع عبور کردید، ۲۰۰ دلار بگیرید.", action: "move_to", value: 24, collect_go: true },
    { text: "پیشروی به میدان ونک", instruction: "اگر از خانه شروع عبور کردید، ۲۰۰ دلار بگیرید.", action: "move_to", value: 11, collect_go: true },
    { text: "پیشروی به نزدیک‌ترین شرکت خدماتی", instruction: "اگر بدون مالک است، می‌توانید آن را بخرید. اگر مالک دارد، تاس بیندازید و ۱۰ برابر عدد تاس به مالک بپردازید.", action: "move_to_nearest", value: "utility" },
    { text: "پیشروی به نزدیک‌ترین راه‌آهن", instruction: "به مالک دو برابر اجاره معمول را بپردازید. اگر بدون مالک است، می‌توانید آن را از بانک بخرید.", action: "move_to_nearest", value: "railroad_double" },
    { text: "پیشروی به نزدیک‌ترین راه‌آهن", instruction: "به مالک دو برابر اجاره معمول را بپردازید. اگر بدون مالک است، می‌توانید آن را از بانک بخرید.", action: "move_to_nearest", value: "railroad_double" },
    { text: "سود سهام", instruction: "بانک به شما ۵۰ دلار سود سهام می‌دهد.", action: "get_money", value: 50 },
    { text: "پیشروی به خیابان نیاوران", instruction: "مستقیماً به این ملک بروید.", action: "move_to", value: 39 },
    { text: "سفری به راه آهن شمال", instruction: "اگر از خانه شروع عبور کردید، ۲۰۰ دلار بگیرید.", action: "move_to", value: 5, collect_go: true },
    { text: "انتخابات", instruction: "شما به عنوان رئیس هیئت مدیره انتخاب شده‌اید. به هر بازیکن ۵۰ دلار بپردازید.", action: "pay_players", value: 50 },
    { text: "وام ساختمانی", instruction: "وام ساختمانی شما سررسید می‌شود. ۱۵۰ دلار دریافت کنید.", action: "get_money", value: 150 },
    { text: "مالیات ناچیز", instruction: "۱۵ دلار بپردازید.", action: "pay_money", value: 15 },
    { text: "جریمه تعمیرات", instruction: "شما باید برای تعمیرات کلی تمام املاک خود هزینه کنید. به ازای هر خانه ۲۵ دلار و به ازای هر هتل ۱۰۰ دلار بپردازید.", action: "street_repairs", house_cost: 25, hotel_cost: 100 }
];

// --- Game State ---
function shuffleDeck(deck) {
    // Creates a copy of the deck to avoid changing the original
    let shuffled = [...deck]; 
    // Fisher-Yates shuffle algorithm
    for (let i = shuffled.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]];
    }
    return shuffled;
}

function promptToDrawCard(deckType) {
    const deckElement = (deckType === 'chance') 
        ? document.getElementById('chance-deck') 
        : document.getElementById('community-chest-deck');

    deckElement.classList.add('active-deck');
    updateAnnouncement('برای کشیدن کارت کلیک کنید...');

    const drawCardHandler = () => {
        clearTimeout(cardDrawTimeout);
        deckElement.classList.remove('active-deck');
        
        let drawnCard;
        if (deckType === 'chance') {
            drawnCard = shuffledChanceCards.shift(); // Get card from top
            shuffledChanceCards.push(drawnCard); // Put it at the bottom
        } else {
            drawnCard = shuffledCommunityChestCards.shift(); // Get card from top
            shuffledCommunityChestCards.push(drawnCard); // Put it at the bottom
        }
        
        showCardModal(drawnCard, deckType);
    };

    deckElement.addEventListener('click', drawCardHandler, { once: true });
    cardDrawTimeout = setTimeout(() => {
        // Remove the event listener to avoid double triggers
        deckElement.removeEventListener('click', drawCardHandler);

        drawCardHandler();
    }, 5000);
}
function showCardModal(card, deckType) {
    // Set text and instruction
    cardModalTitle.textContent = (deckType === 'chance') ? 'کارت شانس' : 'خزانه عمومی';
    cardModalText.textContent = card.text;
    cardModalInstruction.textContent = card.instruction;

    // Reset classes and set new ones for color styling
    cardModalHeader.className = 'card-modal-header';
    cardModalIcon.className = 'card-modal-icon';
    if (deckType === 'chance') {
        cardModalHeader.classList.add('chance-card');
        cardModalIcon.classList.add('chance-card');
        cardModalIcon.textContent = '?';
    } else {
        cardModalHeader.classList.add('chest-card');
        cardModalIcon.classList.add('chest-card');
        cardModalIcon.textContent = '📦';
    }

    // Show the modal
    cardModal.classList.remove('hidden');

    const closeHandler = () => {
        cardModal.classList.add('hidden');
        
        executeCardAction(card);
        
        // --- بخش کلیدی و اصلاح نهایی ---
        // این تابع فقط و فقط در صورتی نوبت را مدیریت می‌کند که کارت از نوع حرکت‌دهنده نباشد
        const movingActions = ['move_to', 'move_relative', 'move_to_nearest', 'go_to_jail'];
        if (!movingActions.includes(card.action)) {
            // اگر بازیکن جفت آورده باشد، نوبت خودش باقی می‌ماند
            if (players[currentPlayerIndex].grantExtraTurn) {
                players[currentPlayerIndex].grantExtraTurn = false;
                diceArea.classList.remove('disabled');
                diceArea.classList.add('dice-area-active');
                updateAnnouncement(`جفت آوردی! دوباره تاس بینداز.`);
            } else {
                // در غیر این صورت، نوبت را به نفر بعدی می‌دهد
                switchTurn();
            }
        }
        // برای کارت‌های حرکت‌دهنده، هیچ کاری انجام نمی‌دهد و مدیریت نوبت را به توابع دیگر می‌سپارد.
    };
    cardModalCloseBtn.addEventListener('click', closeHandler, { once: true });
}

function executeCardAction(card) {
    const player = players[currentPlayerIndex];

    switch (card.action) {
        case 'get_money':
            player.money += card.value;
            updatePlayerMoneyUI(player);
            break;

        case 'pay_money':
            player.money -= card.value;
            updatePlayerMoneyUI(player);
            break;

        case 'go_to_jail':
            goToJail(player); // این تابع خودش نوبت را مدیریت می‌کند
            break;

        case 'get_out_of_jail_free':
            player.getOutOfJailFreeCards++;
            updateJailCardUI(player);
            break;
            
        case 'move_to':
            const passedGo = player.position > card.value && card.collect_go;
            player.position = card.value;
            
            if (passedGo) {
                player.money += 200;
                updatePlayerMoneyUI(player);
                updateAnnouncement(`با عبور از "شروع"، 200$ جایزه گرفتید!`, 2000);
            }

            const destinationSquare = document.getElementById(`space-${player.position}`);
            arrangeTokensInSquare(destinationSquare);
            handleLandingOnSpace(player, 0);
            break;

        case 'move_relative':
            player.position = (player.position + card.value + 40) % 40;
            const relativeDestSquare = document.getElementById(`space-${player.position}`);
            arrangeTokensInSquare(relativeDestSquare);
            handleLandingOnSpace(player, 0);
            break;

        case 'move_to_nearest':
            let destinationIndex;
            let type = card.value;
            
            if (type.includes('railroad')) {
                destinationIndex = findNearestSpace(player.position, 'railroad');
                specialRentMultiplier = 2;
            } else { // 'utility'
                destinationIndex = findNearestSpace(player.position, 'utility');
                // فعلاً منطق این کارت را خالی می‌گذاریم تا بعداً پیاده‌سازی شود
            }

            const stepsToMove = (destinationIndex - player.position + 40) % 40;
            initiateMove(player, stepsToMove, null, false);
            break;
            
        default:
            // برای کارت‌هایی که هنوز پیاده‌سازی نشده‌اند، فقط یک پیام در کنسول نمایش می‌دهیم
            // و هیچ کار دیگری (مخصوصاً تعویض نوبت) انجام نمی‌دهیم.
            console.log(`Action "${card.action}" is not yet implemented.`);
            break;
    }
}
// --- Card Decks Data ---
// (آرایه‌های کارت که قبلا اضافه کردید اینجا قرار دارند)

// --- Game State ---
let gameState = 'token-selection';
let specialRentMultiplier = 1;
let shuffledChanceCards = [];
let shuffledCommunityChestCards = [];
let cardDrawTimeout = null; // برای کنترل تایمر ۵ ثانیه‌ای
// -- پایان بخش اضافه شده --
let players = [];
let currentPlayerIndex = 0;
// ... ادامه کد شما ...
let autoMoveTimer = null;
let auctionState = {}; // <-- این خط جدید را اضافه کنید

// --- DOM Elements ---
const board = document.querySelector('.main-board');
const selectionScreen = document.getElementById('selection-screen');
const gameMessage = document.getElementById('game-message');
const tokenSelectionGrid = document.getElementById('token-selection-grid');
const diceArea = document.getElementById('dice-area');
const dice1El = document.getElementById('dice1');
const dice2El = document.getElementById('dice2');
const playerStatsLeft = document.getElementById('player-stats-left');
const playerStatsRight = document.getElementById('player-stats-right');

// --- Modal Elements ---
const buyModal = document.getElementById('buy-modal');
const buyBtn = document.getElementById('buy-btn');
const passBtn = document.getElementById('pass-btn');
const manageModal = document.getElementById('manage-modal');
const closeManageBtn = document.getElementById('close-manage-btn');
const managePropertiesList = document.getElementById('manage-properties-list');
const cardModal = document.getElementById('card-modal');
const cardModalTitle = document.getElementById('card-modal-title');
const cardModalText = document.getElementById('card-modal-text');
const cardModalInstruction = document.getElementById('card-modal-instruction');
const cardModalIcon = document.getElementById('card-modal-icon');
const cardModalHeader = document.getElementById('card-modal-header');
const cardModalCloseBtn = document.getElementById('card-modal-close-btn');
const jailModal = document.getElementById('jail-modal');
const jailPlayerName = document.getElementById('jail-player-name');
const jailUseCardBtn = document.getElementById('jail-use-card-btn');
const jailPayBtn = document.getElementById('jail-pay-btn');
const jailRollBtn = document.getElementById('jail-roll-btn');
const jailModalCardCount = document.getElementById('jail-modal-card-count');
const jailTurnInfo = document.getElementById('jail-turn-info');
const jailTurnCounter = document.getElementById('jail-turn-counter');
// --- Auction Modal Elements ---
const auctionModal = document.getElementById('auction-modal');
const auctionPropertyNameEl = document.getElementById('auction-property-name');
const highestBidAmountEl = document.getElementById('highest-bid-amount');
const highestBidderNameEl = document.getElementById('highest-bidder-name');
const currentBidderNameEl = document.getElementById('current-bidder-name');
const bidInput = document.getElementById('bid-input');
const submitBidBtn = document.getElementById('submit-bid-btn');
const passAuctionBtn = document.getElementById('pass-auction-btn');
const auctionTokenImg = document.getElementById('auction-token-img');
const auctionColorIndicator = document.getElementById('auction-color-indicator');

// --- Game Functions ---

function updateDiceFace(diceElement, value) {
    diceElement.innerHTML = '';
    diceElement.className = 'dice';
    if (value > 0) {
        diceElement.classList.add(`face-${value}`);
        for (let i = 0; i < value; i++) {
            const dot = document.createElement('div');
            dot.className = 'dot';
            diceElement.appendChild(dot);
        }
    }
}

function updateJailCardUI(player) {
    const indicator = document.getElementById(`jail-card-${player.id}`);
    const countSpan = document.getElementById(`jail-card-count-${player.id}`);
    if (!indicator || !countSpan) return;

    if (player.getOutOfJailFreeCards > 0) {
        countSpan.textContent = player.getOutOfJailFreeCards;
        indicator.style.display = 'flex';
    } else {
        indicator.style.display = 'none';
    }
}

function arrangeTokensInSquare(squareElement) {
    if (!squareElement) return;
    const tokensInSquare = players.filter(p => p.position === parseInt(squareElement.id.split('-')[1]));
    if (tokensInSquare.length === 0) return;

    const squareWidth = squareElement.offsetWidth;
    const tokenSize = tokensInSquare[0].tokenElement.offsetWidth; 
    const gridSize = Math.ceil(Math.sqrt(tokensInSquare.length));
    const padding = 3; 
    const step = (squareWidth - 2 * padding - tokenSize) / (gridSize > 1 ? gridSize - 1 : 1);
    
    tokensInSquare.forEach((player, index) => {
        const row = Math.floor(index / gridSize);
        const col = index % gridSize;
        const posX = padding + col * step;
        const posY = padding + row * step;
        
        const finalLeft = squareElement.offsetLeft + posX;
        const finalTop = squareElement.offsetTop + posY;
        player.tokenElement.style.left = `${finalLeft}px`;
        player.tokenElement.style.top = `${finalTop}px`;
        player.tokenElement.style.transform = `translate(0,0)`;
    });
}

function updateAnnouncement(text, showForMillis) {
    gameMessage.innerHTML = text;
    gameMessage.classList.remove('hidden');
    if(showForMillis) {
        setTimeout(() => {
            gameMessage.classList.add('hidden');
        }, showForMillis);
    }
}

function showBuyModal(player, spaceData) {
    const deedHeader = document.getElementById('deed-header');
    const deedBodyProperty = document.getElementById('deed-body-property');
    const deedBodyUtility = document.getElementById('deed-body-utility');
    const deedBodyRailroad = document.getElementById('deed-body-railroad');
    
    deedHeader.className = 'deed-header';
    deedHeader.innerHTML = `<h3 id="deed-name"></h3>`;

    document.getElementById('deed-name').textContent = spaceData.name;
    document.getElementById('deed-price').textContent = `($${spaceData.price})`;
    document.getElementById('deed-mortgage').textContent = `$${spaceData.mortgage_value}`;
    
    deedBodyProperty.classList.add('hidden');
    deedBodyUtility.classList.add('hidden');
    deedBodyRailroad.classList.add('hidden');

    if (spaceData.type === 'property') {
        deedBodyProperty.classList.remove('hidden');
        deedHeader.classList.add(spaceData.color);
        
        // --- بخش اصلاح شده ---
        // حالا از ایندکس‌های آرایه rent برای پر کردن اطلاعات استفاده می‌کنیم
        document.getElementById('deed-rent').textContent = `$${spaceData.rent[0]}`; // اجاره زمین خالی
        document.getElementById('deed-rent-1').textContent = `$${spaceData.rent[1]}`; // با ۱ خانه
        document.getElementById('deed-rent-2').textContent = `$${spaceData.rent[2]}`; // با ۲ خانه
        document.getElementById('deed-rent-3').textContent = `$${spaceData.rent[3]}`; // با ۳ خانه
        document.getElementById('deed-rent-4').textContent = `$${spaceData.rent[4]}`; // با ۴ خانه
        document.getElementById('deed-rent-hotel').textContent = `$${spaceData.rent[5]}`; // با هتل
        document.getElementById('deed-house-cost').textContent = `$${spaceData.house_cost}`;

    } else if (spaceData.type === 'utility') {
        deedBodyUtility.classList.remove('hidden');
        deedHeader.classList.add('utility');

    } else if (spaceData.type === 'railroad') {
        deedBodyRailroad.classList.remove('hidden');
        deedHeader.classList.add('railroad');
        deedHeader.innerHTML = `<div class="railroad-icon">🚂</div>` + deedHeader.innerHTML;
    }

    if (player.money >= spaceData.price) {
        buyBtn.disabled = false;
    } else {
        buyBtn.disabled = true;
    }
    passBtn.textContent = "مزایده";
    buyModal.classList.remove('hidden');
}

function hideBuyModal() {
    buyModal.classList.add('hidden');
}

/** پنل اطلاعات بازیکنان را در دو ستون چپ و راست می‌سازد */
function setupPlayerStatsUI() {
    playerStatsLeft.innerHTML = '';
    playerStatsRight.innerHTML = '';

    players.forEach((player, index) => {
        const targetContainer = (index < 4) ? playerStatsRight : playerStatsLeft;

        const statDiv = document.createElement('div');
        statDiv.className = 'player-stat';
        statDiv.id = `player-stat-${player.id}`;
        statDiv.innerHTML = `
            <div class="stat-player-info">
                <div class="stat-token-img" style="background-image: url('asset/p${player.tokenId}.png');"></div>
                <div class="player-color-indicator player-color-${player.tokenId}"></div>
                <p id="player-money-${player.id}">$${player.money}</p>
                <div class="jail-card-indicator" id="jail-card-${player.id}" style="display: none;">
                    🔑 <span id="jail-card-count-${player.id}">0</span>
                </div>
            </div>
            <button class="manage-properties-btn" data-player-id="${player.id}">مدیریت املاک</button>
        `;
        targetContainer.appendChild(statDiv);
    });
}

function updatePlayerMoneyUI(player) {
    const moneyEl = document.getElementById(`player-money-${player.id}`);
    if (moneyEl) {
        moneyEl.textContent = `$${player.money}`;
    }
}

/** خانه‌ها و هتل‌ها را روی خانه مربوطه در صفحه بازی نمایش می‌دهد */
function renderHousesOnBoard(property) {
    const spaceElement = document.getElementById(`space-${property.id}`);
    if (!spaceElement) return;

    let oldContainer = spaceElement.querySelector('.house-container');
    if (oldContainer) oldContainer.remove();

    if (property.houses > 0) {
        const houseContainer = document.createElement('div');
        houseContainer.className = 'house-container';

        const houseSVG = `<svg viewBox="0 0 24 24"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>`;

        if (property.houses === 5) {
            const hotelIcon = document.createElement('div');
            hotelIcon.className = 'hotel-icon-board';
            hotelIcon.innerHTML = houseSVG; // از همان آیکون خانه استفاده می‌کنیم، CSS آن را بزرگ و قرمز می‌کند
            houseContainer.appendChild(hotelIcon);
        } else {
            for (let i = 0; i < property.houses; i++) {
                const houseIcon = document.createElement('div');
                houseIcon.className = 'house-icon-board';
                houseIcon.innerHTML = houseSVG;
                houseContainer.appendChild(houseIcon);
            }
        }
        spaceElement.appendChild(houseContainer);
    }
}

function goToJail(player) {
    player.position = 10;
    player.isInJail = true;
    player.jailTurns = 0;
    player.doublesCount = 0;
    player.grantExtraTurn = false; // <-- این خط برای رفع باگ جفت آوردن اضافه شده است
    const jailSquare = document.getElementById('space-10');
    arrangeTokensInSquare(jailSquare);
    updateAnnouncement(`بازیکن ${player.id} به زندان رفت!`, 3000);
    setTimeout(switchTurn, 1500);
}
/** فرآیند مزایده را شروع می‌کند */
function startAuction(property) {
    gameState = 'auction'; // تغییر وضعیت بازی

    // آماده‌سازی وضعیت مزایده
    auctionState = {
        property: property,
        bidders: [...players], // همه بازیکنان در ابتدا شرکت می‌کنند
        highestBid: 0,
        highestBidder: null,
        currentPlayerIndex: currentPlayerIndex // از بازیکنی شروع می‌کنیم که روی خانه فرود آمده
    };

    // به‌روزرسانی اولیه پنجره مزایده
    auctionPropertyNameEl.textContent = property.name;
    highestBidAmountEl.textContent = '$0';
    highestBidderNameEl.textContent = 'هیچکس';

    // نمایش پنجره و شروع نوبت‌دهی
    auctionModal.classList.remove('hidden');
    promptNextBidderInAuction();
}

/** نوبت را به پیشنهاددهنده بعدی در مزایده می‌دهد */
function promptNextBidderInAuction() {
    // ... (منطق پیدا کردن بازیکن بعدی بدون تغییر)
    auctionState.currentPlayerIndex = (auctionState.currentPlayerIndex + 1) % auctionState.bidders.length;
    const currentBidder = auctionState.bidders[auctionState.currentPlayerIndex];

    // --- بخش جدید: نمایش اطلاعات بصری ---
    auctionTokenImg.style.backgroundImage = `url('asset/p${currentBidder.tokenId}.png')`;
    // ابتدا کلاس‌های رنگی قبلی را پاک می‌کنیم
    auctionColorIndicator.className = 'player-color-indicator';
    // کلاس رنگی جدید را اضافه می‌کنیم
    auctionColorIndicator.classList.add(`player-color-${currentBidder.tokenId}`);
    currentBidderNameEl.textContent = `بازیکن ${currentBidder.id}`;

    // --- بخش جدید: منطق افزایش ۵ دلاری ---
    const nextBid = auctionState.highestBid > 0 ? auctionState.highestBid + 5 : 10;
    // مطمئن می‌شویم که پیشنهاد مضربی از ۵ است
    bidInput.value = nextBid - (nextBid % 5);
    bidInput.min = nextBid - (nextBid % 5); // حداقل پیشنهاد را هم مشخص می‌کنیم
}

/** پنجره مدیریت املاک را باز کرده و با اطلاعات بازیکن پر می‌کند */
function openManageModal(player) {
    managePropertiesList.innerHTML = '';
    
    const ownedProperties = boardSpaces.filter(s => s.owner === player.id && (s.type === 'property' || s.type === 'railroad' || s.type === 'utility'));

    if (ownedProperties.length === 0) {
        managePropertiesList.innerHTML = '<p>شما در حال حاضر هیچ ملکی ندارید.</p>';
    } else {
        const propertiesByColor = {};
        ownedProperties.forEach(prop => {
            const group = prop.color || prop.type;
            if (!propertiesByColor[group]) {
                propertiesByColor[group] = [];
            }
            propertiesByColor[group].push(prop);
        });

        for (const group in propertiesByColor) {
            const groupDiv = document.createElement('div');
            groupDiv.className = 'property-group';

            const propertiesInGroup = propertiesByColor[group];
            const hasMonopoly = playerHasMonopoly(player.id, group);
            
            // بررسی وجود خانه در کل گروه
            const housesInGroup = propertiesInGroup.reduce((acc, p) => acc + (p.houses || 0), 0);

            let headerHTML = `<span class="color-name">${group}</span>`;
            if (propertiesInGroup[0].type === 'property') {
                const houseCost = propertiesInGroup[0].house_cost;
                headerHTML += `<span class="house-cost-info">(هزینه هر خانه: $${houseCost})</span>`;
            }
            
            let groupHTML = `<div class="group-header" style="background-color: var(--${group}-color, #888);">${headerHTML}</div>`;

            propertiesInGroup.forEach(prop => {
                const isStreet = prop.type === 'property';
                
                // --- بخش جدید برای دکمه رهن ---
                let mortgageButtonHTML = '';
                if (prop.mortgaged) {
                    const unmortgageCost = Math.round(prop.mortgage_value * 1.1);
                    const canAfford = player.money >= unmortgageCost;
                    mortgageButtonHTML = `
                        <button class="mortgage-action-btn unmortgage-btn" data-action="unmortgage" data-property-id="${prop.id}" ${canAfford ? '' : 'disabled'}>
                            پرداخت رهن ($${unmortgageCost})
                        </button>
                    `;
                } else {
                    // دکمه رهن فقط در صورتی فعال است که هیچ خانه‌ای در گروه نباشد
                    const canMortgage = housesInGroup === 0;
                    mortgageButtonHTML = `
                        <button class="mortgage-action-btn" data-action="mortgage" data-property-id="${prop.id}" ${canMortgage ? '' : 'disabled'}>
                            رهن ($${prop.mortgage_value})
                        </button>
                    `;
                }

                const houseInfo = isStreet ? `(${prop.houses} خانه)` : '';
                const buildButtons = isStreet && hasMonopoly && !prop.mortgaged ? `
                    <div class="build-buttons">
                        <button class="build-btn" data-property-id="${prop.id}">+</button>
                        <button class="sell-btn" data-property-id="${prop.id}">-</button>
                    </div>
                ` : '';

                // کلاس mortgaged برای استایل‌دهی اضافه می‌شود
                const mortgagedClass = prop.mortgaged ? 'mortgaged' : '';

                groupHTML += `
                    <div class="property-entry ${mortgagedClass}" id="prop-entry-${prop.id}">
                        <span>${prop.name} ${houseInfo}</span>
                        <div class="action-buttons">
                            ${buildButtons}
                            ${mortgageButtonHTML}
                        </div>
                    </div>
                `;
            });
            
            groupDiv.innerHTML = groupHTML;
            managePropertiesList.appendChild(groupDiv);
        }
    }

    manageModal.classList.remove('hidden');
}


/** چک می‌کند که آیا یک بازیکن تمام املاک یک رنگ را دارد یا نه */
function playerHasMonopoly(playerId, color) {
    const propertiesOfColor = boardSpaces.filter(s => s.color === color);
    return propertiesOfColor.every(s => s.owner === playerId);
}

function handleLandingOnSpace(player, diceRoll) {
    const spaceIndex = player.position;
    const currentPlayer = players[currentPlayerIndex];
    const spaceData = boardSpaces[spaceIndex];
    let rentAmount = 0;

    if (spaceData.type === 'chance' || spaceData.type === 'chest') {
        promptToDrawCard(spaceData.type);
        return;
    }
    
    if (spaceData.type === 'go-to-jail') {
        updateAnnouncement("مستقیم به زندان برو!");
        setTimeout(() => goToJail(player), 1000);
        return;
    }

    if (spaceData.type === 'property' || spaceData.type === 'railroad' || spaceData.type === 'utility') {
        
        if (spaceData.owner !== null && spaceData.owner !== player.id) {
            const owner = players.find(p => p.id === spaceData.owner);

            if (spaceData.mortgaged) {
                updateAnnouncement(`ملک ${spaceData.name} در رهن است. اجاره‌ای پرداخت نمی‌شود.`);
                setTimeout(() => {
                    if (currentPlayer.grantExtraTurn) {
                        currentPlayer.grantExtraTurn = false;
                        updateAnnouncement(`جفت آوردی! دوباره تاس بیندازید.`);
                        diceArea.classList.remove('disabled');
                        diceArea.classList.add('dice-area-active');
                    } else {
                        switchTurn();
                    }
                }, 2500);
                return; 
            }

            if (spaceData.type === 'property') {
                const hasMonopoly = playerHasMonopoly(owner.id, spaceData.color);
                if (spaceData.houses === 0 && hasMonopoly) {
                    rentAmount = spaceData.rent[0] * 2;
                } else {
                    rentAmount = spaceData.rent[spaceData.houses];
                }
            } else if (spaceData.type === 'railroad') {
                const ownedRailroads = boardSpaces.filter(s => s.type === 'railroad' && s.owner === owner.id).length;
                const rentTiers = [25, 50, 100, 200];
                rentAmount = rentTiers[ownedRailroads - 1];
                
                // --- این بخش اصلاحی است ---
                rentAmount *= specialRentMultiplier; // اعمال ضریب اجاره (مثلاً دو برابر)
                specialRentMultiplier = 1; // بازگرداندن به حالت عادی برای نوبت‌های بعدی
                // --- پایان بخش اصلاحی ---

            } else if (spaceData.type === 'utility') {
                const ownedUtilities = boardSpaces.filter(s => s.type === 'utility' && s.owner === owner.id).length;
                const multiplier = ownedUtilities === 1 ? 4 : 10;
                rentAmount = diceRoll * multiplier;
            }

            if (owner && rentAmount > 0) {
                player.money -= rentAmount;
                owner.money += rentAmount;
                updatePlayerMoneyUI(player);
                updatePlayerMoneyUI(owner);
                updateAnnouncement(`بازیکن ${player.id} مبلغ $${rentAmount} اجاره به بازیکن ${owner.id} پرداخت کرد.`);
                
                setTimeout(() => {
                    if (currentPlayer.grantExtraTurn) {
                        currentPlayer.grantExtraTurn = false;
                        updateAnnouncement(`جفت آوردی! دوباره تاس بیندازай.`);
                        diceArea.classList.remove('disabled');
                        diceArea.classList.add('dice-area-active');
                    } else {
                        switchTurn();
                    }
                }, 2500);
            }

        } else if (spaceData.owner === null) {
            showBuyModal(player, spaceData);
        } else {
            if (currentPlayer.grantExtraTurn) {
                currentPlayer.grantExtraTurn = false;
                updateAnnouncement(`جفت آوردی! دوباره تاس بینداай.`);
                diceArea.classList.remove('disabled');
                diceArea.classList.add('dice-area-active');
            } else {
                switchTurn();
            }
        }
    } else {
        if (spaceData.type === 'tax') {
            const taxAmount = spaceData.amount;
            player.money -= taxAmount;
            updatePlayerMoneyUI(player);
            updateAnnouncement(`بازیکن ${player.id} مبلغ $${taxAmount} مالیات پرداخت کرد.`);
            setTimeout(() => { switchTurn(); }, 2000);
            return;
        }

        if (currentPlayer.grantExtraTurn) {
            currentPlayer.grantExtraTurn = false;
            updateAnnouncement(`جفت آوردی! دوباره تاس بینداай.`);
            diceArea.classList.remove('disabled');
            diceArea.classList.add('dice-area-active');
        } else {
            switchTurn();
        }
    }
}

function switchTurn() {
    // مرحله ۱: پاک کردن افکت‌ها و غیرفعال کردن همه دکمه‌ها
    players.forEach(p => {
        p.tokenElement.classList.remove('active-player-token');
        const statBox = document.getElementById(`player-stat-${p.id}`);
        if (statBox) {
            statBox.classList.remove('active-player-stat');
            for (let i = 1; i <= 8; i++) {
                statBox.classList.remove('player-color-' + i);
            }
            // دکمه مدیریت املاک این بازیکن را پیدا و غیرفعال می‌کنیم
            const manageBtn = statBox.querySelector('.manage-properties-btn');
            if(manageBtn) {
                manageBtn.disabled = true;
            }
        }
    });

    // مرحله ۲: پیدا کردن بازیکن جدید
    currentPlayerIndex = (currentPlayerIndex + 1) % PLAYERS_IN_GAME;
    const newPlayer = players[currentPlayerIndex];

    // مرحله ۳: فعال کردن افکت‌ها و دکمه برای بازیکن جدید
    newPlayer.tokenElement.classList.add('active-player-token');
    const newStatBox = document.getElementById(`player-stat-${newPlayer.id}`);
    if (newStatBox) {
        newStatBox.classList.add('active-player-stat', `player-color-${newPlayer.tokenId}`);
        // دکمه مدیریت املاک بازیکن جدید را پیدا و فعال می‌کنیم
        const newManageBtn = newStatBox.querySelector('.manage-properties-btn');
        if(newManageBtn) {
            newManageBtn.disabled = false;
        }
    }

    // مرحله ۴: اعلام نوبت و فعال کردن تاس
    updateAnnouncement(`نوبت بازیکن ${newPlayer.id} است. تاس بیندازید!`);
    diceArea.classList.remove('disabled');
    diceArea.classList.add('dice-area-active');
}

function initiateMove(player, steps, clickHandler, grantExtraTurn = false) {
    clearTimeout(autoMoveTimer);
    
    const oldPosition = player.position; // موقعیت فعلی را ذخیره می‌کنیم
    const destinationIndex = (oldPosition + steps) % 40;
    
    const destinationSquare = document.getElementById(`space-${destinationIndex}`);
    destinationSquare.classList.remove('destination-square');
    if (clickHandler) {
        destinationSquare.removeEventListener('click', clickHandler);
    }
    
    const oldSquare = document.getElementById(`space-${player.position}`);
    player.position = destinationIndex;
    player.grantExtraTurn = grantExtraTurn;

    // --- منطق جدید برای عبور از خانه شروع ---
    // اگر موقعیت جدید از موقعیت قبلی کمتر باشد، یعنی یک دور زده‌ایم
    if (destinationIndex < oldPosition) {
        player.money += 200;
        updatePlayerMoneyUI(player);
        updateAnnouncement(`بازیکن ${player.id} با عبور از "شروع"، 200$ جایزه گرفت!`, 2000);
    }
    // --- پایان منطق جدید ---

    arrangeTokensInSquare(destinationSquare);

    player.tokenElement.addEventListener('transitionend', () => {
        player.tokenElement.classList.remove('active-player-token');
        arrangeTokensInSquare(oldSquare); 
        handleLandingOnSpace(players[currentPlayerIndex], steps); 
    }, { once: true });
}

function handleTokenClick(event) {
    if (players.length >= PLAYERS_IN_GAME) return;
    const selectionItem = event.currentTarget;
    if (selectionItem.classList.contains('selected')) return;
    
    selectionItem.classList.add('selected');
    const clickedTokenId = parseInt(selectionItem.dataset.id);
    
    const tokenElement = document.createElement('div');
    tokenElement.id = `player-token-${players.length + 1}`;
    tokenElement.className = 'player-token';
    const tokenImage = document.createElement('div');
    tokenImage.className = 'token-image';
    tokenImage.style.backgroundImage = `url('asset/p${clickedTokenId}.png')`;
    tokenElement.appendChild(tokenImage);
    board.appendChild(tokenElement);
    const startRect = selectionItem.getBoundingClientRect();
    const boardRect = board.getBoundingClientRect();
    tokenElement.style.left = `${startRect.left - boardRect.left}px`;
    tokenElement.style.top = `${startRect.top - boardRect.top}px`;
    const playerObject = {
        id: players.length + 1,
        tokenId: clickedTokenId,
        tokenElement: tokenElement,
        position: 0,
        money: 1500,
        orderRoll: 0,
        doublesCount: 0,
        grantExtraTurn: false,
		getOutOfJailFreeCards: 0,
        isInJail: false, // <-- این خط را اضافه کنید
        jailTurns: 0,   // <-- این خط را هم اضافه کنید
    };
    players.push(playerObject);
    document.getElementById('selection-announcement').innerHTML = `بازیکن ${players.length + 1} مهره خود را انتخاب کند...`;
    
    setTimeout(() => {
        const startSquare = document.getElementById('space-0');
        tokenElement.style.width = '5vh';
        tokenElement.style.height = '5vh';
        tokenElement.style.left = `${startSquare.offsetLeft}px`;
        tokenElement.style.top = `${startSquare.offsetTop}px`;
        tokenElement.addEventListener('transitionend', () => arrangeTokensInSquare(startSquare), { once: true });
    }, 50);
    if (players.length === PLAYERS_IN_GAME) {
        setTimeout(startOrderRollPhase, 1000);
    }
}

function startOrderRollPhase() {
    gameState = 'order-roll';
    selectionScreen.classList.add('hidden');
    updateAnnouncement(`برای تعیین نوبت، بازیکن ${players[currentPlayerIndex].id} تاس بیندازد.`);
    diceArea.classList.remove('disabled');
    diceArea.classList.add('dice-area-active');
}

function determinePlayOrder() {
    players.sort((a, b) => b.orderRoll - a.orderRoll);
    let orderString = "ترتیب نهایی بازی: ";
    players.forEach((p, index) => {
        orderString += `بازیکن ${p.id}` + (index < players.length - 1 ? "، سپس " : "");
    });
    updateAnnouncement(orderString);
    
    setTimeout(() => {
        gameState = 'game-play';
        currentPlayerIndex = -1;
        setupPlayerStatsUI();
        switchTurn(); 
    }, 4000);
}

function handleDiceClick() {
    if (diceArea.classList.contains('disabled')) return;
    const player = players[currentPlayerIndex];
    diceArea.classList.remove('dice-area-active');

    if (gameState === 'order-roll') {
        diceArea.classList.add('disabled');
        dice2El.style.display = 'none';
        const roll = Math.floor(Math.random() * 6) + 1;
        updateDiceFace(dice1El, roll);
        player.orderRoll = roll;
        updateAnnouncement(`بازیکن ${player.id} عدد ${roll} آورد.`, 1500);
        currentPlayerIndex++;
        setTimeout(() => {
            if (currentPlayerIndex < players.length) {
                updateAnnouncement(`نوبت بازیکن ${players[currentPlayerIndex].id} برای انداختن تاس.`);
                diceArea.classList.remove('disabled');
                diceArea.classList.add('dice-area-active');
            } else {
                updateAnnouncement('در حال تعیین ترتیب بازی...');
                setTimeout(determinePlayOrder, 1500);
            }
        }, 1500);
    } else if (gameState === 'game-play') {
        if (player.isInJail) {
            showJailOptions(player);
            return;
        }

        diceArea.classList.add('disabled');
        dice2El.style.display = 'grid';
        const roll1 = Math.floor(Math.random() * 6) + 1;
        const roll2 = Math.floor(Math.random() * 6) + 1;
        updateDiceFace(dice1El, roll1);
        updateDiceFace(dice2El, roll2);
        
        const rolledDoubles = roll1 === roll2;
        
        if (rolledDoubles) {
            player.doublesCount++;
            if (player.doublesCount === 3) {
                updateAnnouncement("جفت سوم! مستقیم به زندان برو.");
                goToJail(player);
                return;
            }
            updateAnnouncement(`جفت ${player.doublesCount}! حرکت کن و دوباره تاس بینداز.`);
        } else {
            player.doublesCount = 0;
            updateAnnouncement(`شما ${roll1 + roll2} آوردید. حرکت کنید.`);
        }
        
        const totalMove = roll1 + roll2;
        const destinationIndex = (player.position + totalMove) % 40;
        const destinationSquare = document.getElementById(`space-${destinationIndex}`);
        destinationSquare.classList.add('destination-square');
        const moveHandler = () => initiateMove(player, totalMove, moveHandler, rolledDoubles);
        
        destinationSquare.addEventListener('click', moveHandler, { once: true });
        autoMoveTimer = setTimeout(() => {
            initiateMove(player, totalMove, moveHandler, rolledDoubles);
        }, 5000);
    
    } else if (gameState === 'jail-roll') {
        diceArea.classList.add('disabled');
        dice2El.style.display = 'grid';
        
        const roll1 = Math.floor(Math.random() * 6) + 1;
        const roll2 = Math.floor(Math.random() * 6) + 1;
        updateDiceFace(dice1El, roll1);
        updateDiceFace(dice2El, roll2);

        if (roll1 === roll2) {
            updateAnnouncement(`جفت ${roll1} آوردی! از زندان آزاد شدی.`);
            setTimeout(() => {
                player.isInJail = false;
                player.jailTurns = 0;
                gameState = 'game-play';
                updateAnnouncement(`برای حرکت ${roll1 + roll2} خانه کلیک کن.`);
                initiateMove(player, roll1 + roll2, null, false);
            }, 2000);
        } else {
            player.jailTurns++;
            updateAnnouncement(`جفت نیاوردی (${roll1} و ${roll2}). نوبت تمام شد.`);
            if (player.jailTurns >= 3) {
                setTimeout(() => {
                    updateAnnouncement(`سه نوبت تمام شد! باید 50$ جریمه بپردازی.`);
                    player.money -= 50;
                    updatePlayerMoneyUI(player);
                    getOutOfJail(player);
                }, 2000);
            } else {
                // ***** این خط کد اصلاحی است *****
                gameState = 'game-play'; // وضعیت بازی را به حالت عادی برمی‌گردانیم
                // *******************************
                setTimeout(switchTurn, 2000);
            }
        }
    }
}

function initializeGame() {
    const selectionAnnouncement = document.createElement('div');
    selectionAnnouncement.id = 'selection-announcement';
    Object.assign(selectionAnnouncement.style, {
        fontSize: '3vh', fontWeight: 'bold', color: '#1a562a', textAlign: 'center', marginBottom: '4vh'
    });
    selectionAnnouncement.innerHTML = `بازی ${PLAYERS_IN_GAME} نفره است. <br> بازیکن 1، مهره خود را انتخاب کند.`;
    selectionScreen.prepend(selectionAnnouncement);
    diceArea.classList.add('disabled');
    updateDiceFace(dice1El, 0);
    updateDiceFace(dice2El, 0);
    for (let i = 1; i <= 8; i++) {
        const selectionItem = document.createElement('div');
        selectionItem.id = `p-token-${i}`;
        selectionItem.className = 'player-token selectable';
        selectionItem.style.position = 'relative';
        const tokenImage = document.createElement('div');
        tokenImage.className = 'token-image';
        selectionItem.appendChild(tokenImage);
        selectionItem.dataset.id = i;
        selectionItem.addEventListener('click', handleTokenClick);
        tokenSelectionGrid.appendChild(selectionItem);
    }
	shuffledChanceCards = shuffleDeck(chanceCards);
shuffledCommunityChestCards = shuffleDeck(communityChestCards);
}

// --- Event Listeners ---
buyBtn.addEventListener('click', () => {
    const currentPlayer = players[currentPlayerIndex];
    const currentPosition = currentPlayer.position;
    
    if (currentPosition < boardSpaces.length) {
        const spaceData = boardSpaces[currentPosition];

        if (currentPlayer.money >= spaceData.price) {
            currentPlayer.money -= spaceData.price;
            spaceData.owner = currentPlayer.id;
            updatePlayerMoneyUI(currentPlayer);
            // updatePlayerPropertiesUI(currentPlayer); // <--- این خط مشکل‌ساز حذف شد

            const spaceElement = document.getElementById(`space-${currentPosition}`);
            if(spaceElement) {
                const existingMarker = spaceElement.querySelector('.owner-marker');
                if (existingMarker) existingMarker.remove();
                
                const ownerMarker = document.createElement('div');
                ownerMarker.className = `owner-marker player-color-${currentPlayer.tokenId}`;
                spaceElement.appendChild(ownerMarker);
            }
        }
    }
    
    hideBuyModal();
    if (currentPlayer.grantExtraTurn) {
        currentPlayer.grantExtraTurn = false;
        diceArea.classList.remove('disabled');
        diceArea.classList.add('dice-area-active');
    } else {
        switchTurn();
    }
});

passBtn.addEventListener('click', () => {
    const currentPlayer = players[currentPlayerIndex];
    const property = boardSpaces[currentPlayer.position];

    hideBuyModal();
    startAuction(property); // به جای رد کردن نوبت، مزایده را شروع می‌کنیم
});

diceArea.addEventListener('click', handleDiceClick);

closeManageBtn.addEventListener('click', () => {
    manageModal.classList.add('hidden');
});

// --- Event Listeners for Auction Modal ---
submitBidBtn.addEventListener('click', () => {
    const currentBidder = auctionState.bidders[auctionState.currentPlayerIndex];
    const bidAmount = parseInt(bidInput.value);

    // --- بررسی قوانین پیشنهاد ---
    if (bidAmount <= auctionState.highestBid) {
        alert(`پیشنهاد شما باید بیشتر از $${auctionState.highestBid} باشد.`);
        return;
    }
    if (bidAmount > currentBidder.money) {
        alert("شما پول کافی برای این پیشنهاد ندارید!");
        return;
    }
    if (bidAmount % 5 !== 0) {
        alert("مبلغ پیشنهادی باید مضربی از ۵ باشد.");
        return;
    }

    // --- ثبت پیشنهاد جدید ---
    auctionState.highestBid = bidAmount;
    auctionState.highestBidder = currentBidder;

    // به‌روزرسانی نمایش بالاترین پیشنهاد
    highestBidAmountEl.textContent = `$${bidAmount}`;
    highestBidderNameEl.textContent = `بازیکن ${currentBidder.id}`;

    // رفتن به نفر بعدی
    promptNextBidderInAuction();
});

passAuctionBtn.addEventListener('click', () => {
    // حذف بازیکن فعلی از لیست شرکت‌کنندگان
    auctionState.bidders.splice(auctionState.currentPlayerIndex, 1);

    // اگر فقط یک نفر باقی ماند، او برنده است
    if (auctionState.bidders.length === 1) {
        const winner = auctionState.bidders[0];
        const property = auctionState.property;

        updateAnnouncement(`مزایده تمام شد! بازیکن ${winner.id} ملک ${property.name} را با قیمت $${auctionState.highestBid} خرید.`);

        // انتقال پول و ملک
        winner.money -= auctionState.highestBid;
        property.owner = winner.id;
        updatePlayerMoneyUI(winner);

        // اضافه کردن نشانگر مالکیت
        const spaceElement = document.getElementById(`space-${property.id}`);
        if(spaceElement) {
            const ownerMarker = document.createElement('div');
            ownerMarker.className = `owner-marker player-color-${winner.tokenId}`;
            spaceElement.appendChild(ownerMarker);
        }

        // بازگشت به بازی
        auctionModal.classList.add('hidden');
        gameState = 'game-play';
        // نوبت را به بازیکن بعدی می‌دهیم
        switchTurn(); 
    } else {
        // اگر بیش از یک نفر باقی مانده، به نفر بعدی برو
        // چون یک نفر حذف شده، ایندکس را یکی به عقب برمی‌گردانیم
        auctionState.currentPlayerIndex--;
        promptNextBidderInAuction();
    }
});

// --- Event Listener for Manage Modal Buttons (Build, Sell, Mortgage) ---
managePropertiesList.addEventListener('click', (event) => {
    const target = event.target;
    const propertyId = parseInt(target.dataset.propertyId);
    if (!propertyId && !target.classList.contains('cancel-btn')) return; // اگر روی دکمه‌ای که شناسه ملک ندارد کلیک نشده، خارج شو

    const property = boardSpaces.find(s => s.id === propertyId);
    const player = players.find(p => p.id === property?.owner);
    if (!player && !target.classList.contains('cancel-btn')) return;

    const action = target.dataset.action;

    // --- منطق نمایش دکمه‌های تایید/لغو ---
    if (action === 'mortgage' || action === 'unmortgage') {
        const entryElement = document.getElementById(`prop-entry-${propertyId}`);
        const actionButtonsDiv = entryElement.querySelector('.action-buttons');
        
        // محتوای فعلی دکمه‌ها را ذخیره می‌کنیم تا در صورت لغو، بازگردانیم
        const originalButtonsHTML = actionButtonsDiv.innerHTML;

        let confirmationHTML = '';
        if (action === 'mortgage') {
            confirmationHTML = `
                <span>مطمئن هستید؟</span>
                <button class="mortgage-action-btn confirm-btn" data-action="confirm-mortgage" data-property-id="${propertyId}">تأیید</button>
                <button class="mortgage-action-btn cancel-btn" data-original-html='${originalButtonsHTML}'>لغو</button>
            `;
        } else { // action === 'unmortgage'
            const cost = Math.round(property.mortgage_value * 1.1);
            confirmationHTML = `
                <span>پرداخت $${cost}؟</span>
                <button class="mortgage-action-btn confirm-btn" data-action="confirm-unmortgage" data-property-id="${propertyId}">تأیید</button>
                <button class="mortgage-action-btn cancel-btn" data-original-html='${originalButtonsHTML}'>لغو</button>
            `;
        }
        actionButtonsDiv.innerHTML = confirmationHTML;
        return; // منتظر کلیک بعدی کاربر می‌مانیم
    }

    // --- منطق دکمه لغو ---
    if (target.classList.contains('cancel-btn')) {
        // برای لغو، فقط کافیست پنجره را دوباره رفرش کنیم
        const currentPlayerId = document.querySelector('.active-player-stat').id.split('-')[2];
        const currentPlayer = players.find(p => p.id == currentPlayerId);
        if(currentPlayer) openManageModal(currentPlayer);
        return;
    }

    // --- منطق تایید نهایی برای رهن و پرداخت رهن ---
    if (action === 'confirm-mortgage') {
        player.money += property.mortgage_value;
        property.mortgaged = true;
        updatePlayerMoneyUI(player);
        // اضافه کردن یک نشانگر بصری روی خانه در صفحه بازی
        const spaceElement = document.getElementById(`space-${property.id}`);
        if(spaceElement) spaceElement.style.opacity = '0.6';

    } else if (action === 'confirm-unmortgage') {
        const cost = Math.round(property.mortgage_value * 1.1);
        player.money -= cost;
        property.mortgaged = false;
        updatePlayerMoneyUI(player);
        // حذف نشانگر بصری از روی خانه
        const spaceElement = document.getElementById(`space-${property.id}`);
        if(spaceElement) spaceElement.style.opacity = '1';
    }

    // --- منطق ساخت و فروش خانه (بدون تغییر باقی مانده است) ---
    else if (target.classList.contains('build-btn')) {
        if (player.money < property.house_cost) {
            alert("شما پول کافی برای ساخت خانه ندارید!");
            return;
        }
        if (property.houses >= 5) {
            alert("شما به حداکثر تعداد ساختمان (هتل) رسیده‌اید.");
            return;
        }
        const propertiesInGroup = boardSpaces.filter(s => s.color === property.color);
        const canBuild = propertiesInGroup.every(p => property.houses <= p.houses);
        if (!canBuild) {
            alert("شما باید به صورت یکنواخت خانه بسازید.");
            return;
        }
        player.money -= property.house_cost;
        property.houses++;
        updatePlayerMoneyUI(player);
        renderHousesOnBoard(property);

    } else if (target.classList.contains('sell-btn')) {
        if (property.houses <= 0) {
            alert("هیچ خانه‌ای برای فروش وجود ندارد.");
            return;
        }
        const propertiesInGroup = boardSpaces.filter(s => s.color === property.color);
        const canSell = propertiesInGroup.every(p => property.houses >= p.houses);
        if (!canSell) {
            alert("شما باید به صورت یکنواخت خانه بفروشید.");
            return;
        }
        player.money += property.house_cost / 2;
        property.houses--;
        updatePlayerMoneyUI(player);
        renderHousesOnBoard(property);
    }
    
    // در نهایت، پنجره را با اطلاعات جدید رفرش می‌کنیم
    if (player) {
        openManageModal(player);
    }
});

// تابع مدیریت کلیک روی دکمه‌های "مدیریت املاک"
function handleManageButtonClick(event) {
    if (event.target.classList.contains('manage-properties-btn')) {
        const playerId = parseInt(event.target.dataset.playerId);
        const player = players.find(p => p.id === playerId);
        if (player) {
            openManageModal(player);
        }
    }
}

// فعال کردن دکمه‌ها برای هر دو ستون
playerStatsLeft.addEventListener('click', handleManageButtonClick);
playerStatsRight.addEventListener('click', handleManageButtonClick);

/**
 * نزدیک‌ترین خانه از یک نوع خاص (مثلاً راه‌آهن) را پیدا می‌کند
 * @param {number} currentPosition - موقعیت فعلی بازیکن
 * @param {string} type - نوع ملک مورد نظر ('railroad' یا 'utility')
 * @returns {number} - ایندکس خانه مقصد
 */
function findNearestSpace(currentPosition, type) {
    const spaceIndexes = [];
    boardSpaces.forEach((space, index) => {
        if (space.type === type) {
            spaceIndexes.push(index);
        }
    });

    for (const index of spaceIndexes) {
        if (index > currentPosition) {
            return index; // اولین مورد بعدی را برمی‌گرداند
        }
    }
    return spaceIndexes[0]; // اگر از همه عبور کرده بود، به اولین مورد در صفحه برمی‌گردد
}

// --- Jail Logic ---

// این تابع پنجره گزینه‌های زندان را باز و مدیریت می‌کند
function showJailOptions(player) {
    gameState = 'in-jail-turn'; // وضعیت بازی را تغییر می‌دهیم
    
    // --- خط جدید برای نمایش نوبت ---
    jailTurnCounter.textContent = `نوبت ${player.jailTurns + 1} از 3`;
    // ---
    
    jailPlayerName.textContent = `بازیکن ${player.id}`;
    
    // دکمه کارت آزادی را فقط اگر بازیکن کارت داشته باشد فعال می‌کنیم
    if (player.getOutOfJailFreeCards > 0) {
        jailUseCardBtn.disabled = false;
        jailModalCardCount.textContent = player.getOutOfJailFreeCards;
    } else {
        jailUseCardBtn.disabled = true;
        jailModalCardCount.textContent = 0;
    }

    // دکمه پرداخت جریمه را فقط اگر پول کافی داشته باشد فعال می‌کنیم
    jailPayBtn.disabled = player.money < 50;

    // اطلاعات مربوط به نوبت را نمایش می‌دهیم
    jailTurnInfo.textContent = `شما ${player.jailTurns} نوبت تلاش ناموفق برای خروج داشته‌اید.`;
    
    jailModal.classList.remove('hidden');
}

// تابع برای خروج از زندان (برای جلوگیری از تکرار کد)
function getOutOfJail(player) {
    player.isInJail = false;
    player.jailTurns = 0;
    jailModal.classList.add('hidden');
    updateAnnouncement(`بازیکن ${player.id} از زندان آزاد شد! برای حرکت تاس بیندازید.`);
    
    // تاس را برای حرکت عادی فعال می‌کنیم
    diceArea.classList.remove('disabled');
    diceArea.classList.add('dice-area-active');
    gameState = 'game-play'; // وضعیت بازی را به حالت عادی برمی‌گردانیم
}

// Event Listener برای دکمه "استفاده از کارت"
jailUseCardBtn.addEventListener('click', () => {
    const player = players[currentPlayerIndex];
    if (player.getOutOfJailFreeCards > 0) {
        player.getOutOfJailFreeCards--;
        updateJailCardUI(player);
        getOutOfJail(player);
    }
});

// Event Listener برای دکمه "پرداخت جریمه"
jailPayBtn.addEventListener('click', () => {
    const player = players[currentPlayerIndex];
    if (player.money >= 50) {
        player.money -= 50;
        updatePlayerMoneyUI(player);
        getOutOfJail(player);
    }
});

// Event Listener برای دکمه "تاس ریختن برای جفت"
jailRollBtn.addEventListener('click', () => {
    // این تابع دیگر خودش تاس نمی‌ریزد.
    // فقط پنجره را می‌بندد و تاس را برای کلیک بازیکن فعال می‌کند.
    jailModal.classList.add('hidden');
    diceArea.classList.remove('disabled');
    diceArea.classList.add('dice-area-active');
    updateAnnouncement('برای جفت آوردن، روی تاس کلیک کنید...');
    gameState = 'jail-roll'; // وضعیت بازی را برای تاس ریختن در زندان تنظیم می‌کنیم
});

// --- Game Initialization ---
initializeGame();